import { MeetingModel, MeetingStatus } from "../database/models/meeting.model.js";
import { MeetingFilterEnum } from "../enums/meeting.enum.js";
import {
  EventModel,
  EventLocationEnumType,
} from "../database/models/event.model.js";
import {
  IntegrationModel,
  IntegrationAppTypeEnum,
} from "../database/models/integration.model.js";
import { BadRequestException, NotFoundException } from "../utils/app-error.js";
import { validateGoogleToken } from "./integration.service.js";
import { googleOAuth2Client } from "../config/oauth.config.js";
import { google } from "googleapis";

export const getUserMeetingsService = async (
  userId,
  filter
) => {
  const where = { user: userId };
  const now = new Date();

  if (filter === MeetingFilterEnum.UPCOMING) {
    where.status = MeetingStatus.SCHEDULED;
    where.startTime = { $gt: now };
  } else if (filter === MeetingFilterEnum.PAST) {
    where.status = MeetingStatus.SCHEDULED;
    where.startTime = { $lt: now };
  } else if (filter === MeetingFilterEnum.CANCELLED) {
    where.status = MeetingStatus.CANCELLED;
  } else {
    where.status = MeetingStatus.SCHEDULED;
    where.startTime = { $gt: now };
  }

  const meetings = await MeetingModel.find(where)
    .populate("event")
    .sort({ startTime: 1 });

  return meetings || [];
};

export const createMeetBookingForGuestService = async (
  createMeetingDto
) => {
  const { eventId, guestEmail, guestName, additionalInfo } = createMeetingDto;
  const startTime = new Date(createMeetingDto.startTime);
  const endTime = new Date(createMeetingDto.endTime);

  const event = await EventModel.findOne({
    _id: eventId,
    isPrivate: false,
  }).populate("user");

  if (!event) throw new NotFoundException("Event not found");

  if (!Object.values(EventLocationEnumType).includes(event.locationType)) {
    throw new BadRequestException("Invalid location type");
  }

  const meetIntegration = await IntegrationModel.findOne({
    user: event.user._id,
    app_type: IntegrationAppTypeEnum[event.locationType],
  });

  if (!meetIntegration)
    throw new BadRequestException("No video conferencing integration found");

  let meetLink = "";
  let calendarEventId = "";
  let calendarAppType = "";

  if (event.locationType === EventLocationEnumType.GOOGLE_MEET_AND_CALENDAR) {
    const { calendarType, calendar } = await getCalendarClient(
      meetIntegration.app_type,
      meetIntegration.access_token,
      meetIntegration.refresh_token,
      meetIntegration.expiry_date
    );
    const response = await calendar.events.insert({
      calendarId: "primary",
      conferenceDataVersion: 1,
      requestBody: {
        summary: `${guestName} - ${event.title}`,
        description: additionalInfo,
        start: { dateTime: startTime.toISOString() },
        end: { dateTime: endTime.toISOString() },
        attendees: [{ email: guestEmail }, { email: event.user.email }],
        conferenceData: {
          createRequest: {
            requestId: `${event.id}-${Date.now()}`,
          },
        },
      },
    });

    meetLink = response.data.hangoutLink;
    calendarEventId = response.data.id;
    calendarAppType = calendarType;
  }

  const meeting = await MeetingModel.create({
    event: event._id,
    user: event.user._id,
    guestName,
    guestEmail,
    additionalInfo,
    startTime,
    endTime,
    meetLink: meetLink,
    calendarEventId: calendarEventId,
    calendarAppType: calendarAppType,
  });

  return {
    meetLink,
    meeting,
  };
};

export const cancelMeetingService = async (meetingId) => {
  const meeting = await MeetingModel.findById(meetingId).populate({
    path: "event",
    populate: { path: "user" },
  });

  if (!meeting) throw new NotFoundException("Meeting not found");

  try {
    const appTypeKey = meeting.calendarAppType;
    
    const calendarIntegration = await IntegrationModel.findOne({
      user: meeting.event.user._id,
      app_type: meeting.calendarAppType,
    });

    if (calendarIntegration) {
      const { calendar, calendarType } = await getCalendarClient(
        calendarIntegration.app_type,
        calendarIntegration.access_token,
        calendarIntegration.refresh_token,
        calendarIntegration.expiry_date
      );
      switch (calendarType) {
        case IntegrationAppTypeEnum.GOOGLE_MEET_AND_CALENDAR:
          await calendar.events.delete({
            calendarId: "primary",
            eventId: meeting.calendarEventId,
          });
          break;
        default:
          throw new BadRequestException(
            `Unsupported calendar provider: ${calendarType}`
          );
      }
    }
  } catch (error) {
    throw new BadRequestException("Failed to delete event from calendar");
  }

  meeting.status = MeetingStatus.CANCELLED;
  await meeting.save();
  return { success: true };
};

async function getCalendarClient(
  appType,
  access_token,
  refresh_token, 
  expiry_date 
) {
  switch (appType) {
    case IntegrationAppTypeEnum.GOOGLE_MEET_AND_CALENDAR:
      const validToken = await validateGoogleToken(
        access_token,
        refresh_token || "",
        expiry_date || null
      );
      googleOAuth2Client.setCredentials({ access_token: validToken });
      const calendar = google.calendar({
        version: "v3",
        auth: googleOAuth2Client,
      });
      return {
        calendar,
        calendarType: IntegrationAppTypeEnum.GOOGLE_MEET_AND_CALENDAR,
      };
    default:
      throw new BadRequestException(
        `Unsupported Calendar provider: ${appType}`
      );
  }
}
