import {
  IntegrationModel,
  IntegrationAppTypeEnum,
  IntegrationCategoryEnum,
  IntegrationProviderEnum,
} from "../database/models/integration.model.js";
import { BadRequestException } from "../utils/app-error.js";
import { googleOAuth2Client } from "../config/oauth.config.js";
import { encodeState } from "../utils/helper.js";

const appTypeToProviderMap = {
  [IntegrationAppTypeEnum.GOOGLE_MEET_AND_CALENDAR]:
    IntegrationProviderEnum.GOOGLE,
  [IntegrationAppTypeEnum.ZOOM_MEETING]: IntegrationProviderEnum.ZOOM,
  [IntegrationAppTypeEnum.OUTLOOK_CALENDAR]: IntegrationProviderEnum.MICROSOFT,
  [IntegrationAppTypeEnum.SALESFORCE]: IntegrationProviderEnum.SALESFORCE,
  [IntegrationAppTypeEnum.HUBSPOT]: IntegrationProviderEnum.HUBSPOT,
  [IntegrationAppTypeEnum.ZAPIER]: IntegrationProviderEnum.ZAPIER,
};

const appTypeToCategoryMap = {
  [IntegrationAppTypeEnum.GOOGLE_MEET_AND_CALENDAR]:
    IntegrationCategoryEnum.CALENDAR_AND_VIDEO_CONFERENCING,
  [IntegrationAppTypeEnum.ZOOM_MEETING]:
    IntegrationCategoryEnum.VIDEO_CONFERENCING,
  [IntegrationAppTypeEnum.OUTLOOK_CALENDAR]: IntegrationCategoryEnum.CALENDAR,
  [IntegrationAppTypeEnum.SALESFORCE]: IntegrationCategoryEnum.CRM,
  [IntegrationAppTypeEnum.HUBSPOT]: IntegrationCategoryEnum.CRM,
  [IntegrationAppTypeEnum.ZAPIER]: IntegrationCategoryEnum.AUTOMATION,
};

const appTypeToTitleMap = {
  [IntegrationAppTypeEnum.GOOGLE_MEET_AND_CALENDAR]: "Google Meet & Calendar",
  [IntegrationAppTypeEnum.ZOOM_MEETING]: "Zoom",
  [IntegrationAppTypeEnum.OUTLOOK_CALENDAR]: "Outlook Calendar",
  [IntegrationAppTypeEnum.SALESFORCE]: "Salesforce",
  [IntegrationAppTypeEnum.HUBSPOT]: "HubSpot",
  [IntegrationAppTypeEnum.ZAPIER]: "Zapier",
};

export const getUserIntegrationsService = async (userId) => {
  const userIntegrations = await IntegrationModel.find({
    user: userId,
  });

  const connectedMap = new Map(
    userIntegrations.map((integration) => [integration.app_type, true])
  );

  return Object.values(IntegrationAppTypeEnum).flatMap((appType) => {
    return {
      provider: appTypeToProviderMap[appType],
      title: appTypeToTitleMap[appType],
      app_type: appType,
      category: appTypeToCategoryMap[appType],
      isConnected: connectedMap.has(appType) || false,
    };
  });
};

export const checkIntegrationService = async (
  userId,
  appType
) => {
  const integration = await IntegrationModel.findOne({
    user: userId,
    app_type: appType,
  });

  if (!integration) {
    return false;
  }

  return true;
};

export const connectAppService = async (
  userId,
  appType
) => {
  const state = encodeState({ userId, appType });

  let authUrl;

  switch (appType) {
    case IntegrationAppTypeEnum.GOOGLE_MEET_AND_CALENDAR:
      authUrl = googleOAuth2Client.generateAuthUrl({
        access_type: "offline",
        scope: ["https://www.googleapis.com/auth/calendar.events"],
        prompt: "consent",
        state,
      });
      break;
    default:
      throw new BadRequestException("Unsupported app type");
  }

  return { url: authUrl };
};

export const createIntegrationService = async (data) => {
  const existingIntegration = await IntegrationModel.findOne({
    user: data.userId,
    app_type: data.app_type,
  });

  if (existingIntegration) {
    throw new BadRequestException(`${data.app_type} already connected`);
  }

  const integration = await IntegrationModel.create({
    provider: data.provider,
    category: data.category,
    app_type: data.app_type,
    access_token: data.access_token,
    refresh_token: data.refresh_token,
    expiry_date: data.expiry_date,
    metadata: data.metadata,
    user: data.userId,
    isConnected: true,
  });

  return integration;
};

export const validateGoogleToken = async (
  accessToken,
  refreshToken,
  expiryDate
) => {
  if (expiryDate === null || Date.now() >= expiryDate) {
    googleOAuth2Client.setCredentials({
      refresh_token: refreshToken,
    });
    const { credentials } = await googleOAuth2Client.refreshAccessToken();
    return credentials.access_token;
  }

  return accessToken;
};
