import {
  EventModel,
  EventLocationEnumType,
} from "../database/models/event.model.js";
import { UserModel } from "../database/models/user.model.js";
import { MeetingModel } from "../database/models/meeting.model.js";
import { BadRequestException, NotFoundException } from "../utils/app-error.js";
import { slugify } from "../utils/helper.js";

export const createEventService = async (
  userId,
  createEventDto
) => {
  if (
    !Object.values(EventLocationEnumType)?.includes(createEventDto.locationType)
  ) {
    throw new BadRequestException("Invalid location type");
  }

  const slug = slugify(createEventDto.title);

  const event = await EventModel.create({
    ...createEventDto,
    slug,
    user: userId,
  });

  return event;
};

export const getUserEventsService = async (userId) => {
  const user = await UserModel.findById(userId);

  if (!user) {
    throw new NotFoundException("User not found");
  }

  const events = await EventModel.find({ user: userId }).sort({ createdAt: -1 });

  // Attach meeting count
  const eventsWithCount = await Promise.all(
    events.map(async (event) => {
      const count = await MeetingModel.countDocuments({ event: event._id });
      const eventObj = event.toObject();
      eventObj._count = { meetings: count };
      return eventObj;
    })
  );

  return {
    events: eventsWithCount,
    username: user.username,
  };
};

export const toggleEventPrivacyService = async (
  userId,
  eventId
) => {
  const event = await EventModel.findOne({ _id: eventId, user: userId });

  if (!event) {
    throw new NotFoundException("Event not found");
  }

  event.isPrivate = !event.isPrivate;
  await event.save();

  return event;
};

// Public Events
export const getPublicEventsByUsernameService = async (username) => {
  const user = await UserModel.findOne({ username });

  if (!user) {
    throw new NotFoundException("User not found");
  }

  const events = await EventModel.find({ user: user._id, isPrivate: false })
    .sort({ createdAt: -1 })
    .select("id title description slug duration locationType createdAt");

  return {
    user: {
      name: user.name,
      username: username,
      imageUrl: user.imageUrl,
    },
    events,
  };
};

export const getPublicEventByUsernameAndSlugService = async (
  userNameAndSlugDto
) => {
  const { username, slug } = userNameAndSlugDto;
  
  const user = await UserModel.findOne({ username });
  
  if (!user) return null;

  const event = await EventModel.findOne({
    user: user._id,
    slug: slug,
    isPrivate: false,
  }).select("id title description slug duration locationType");

  if (!event) return null;

  const eventObj = event.toObject();
  eventObj.user = {
    id: user.id,
    name: user.name,
    imageUrl: user.imageUrl,
  };

  return eventObj;
};

export const deleteEventService = async (userId, eventId) => {
  const event = await EventModel.findOneAndDelete({ _id: eventId, user: userId });

  if (!event) {
    throw new NotFoundException("Event not found");
  }

  return { success: true };
};

export const updateEventService = async (userId, eventId, updateEventDto) => {
  const event = await EventModel.findOne({ _id: eventId, user: userId });

  if (!event) {
    throw new NotFoundException("Event not found");
  }

  if (updateEventDto.title) {
    updateEventDto.slug = slugify(updateEventDto.title);
  }

  Object.assign(event, updateEventDto);
  await event.save();

  return event;
};
