import { v4 as uuidv4 } from "uuid";
import { UserModel } from "../database/models/user.model.js";
import {
  BadRequestException,
  NotFoundException,
  UnauthorizedException,
} from "../utils/app-error.js";
import { AvailabilityModel, DayOfWeekEnum } from "../database/models/availability.model.js";
import { signJwtToken } from "../utils/jwt.js";

export const registerService = async (registerDto) => {
  const existingUser = await UserModel.findOne({ email: registerDto.email });

  if (existingUser) {
    throw new BadRequestException("User already exists");
  }

  const username = await generateUsername(registerDto.name);
  
  const user = new UserModel({
    ...registerDto,
    username,
  });

  const availability = new AvailabilityModel({
    user: user._id,
    timeGap: 30,
    days: Object.values(DayOfWeekEnum).map((day) => {
      return {
        day: day,
        startTime: new Date(`2025-03-01T09:00:00Z`), //9:00
        endTime: new Date(`2025-03-01T17:00:00Z`), //5:00pm
        isAvailable:
          day !== DayOfWeekEnum.SUNDAY && day !== DayOfWeekEnum.SATURDAY,
      };
    }),
  });

  await user.save();
  await availability.save();

  return { user: user.omitPassword() };
};

export const loginService = async (loginDto) => {
  const user = await UserModel.findOne({ email: loginDto.email });

  if (!user) {
    throw new NotFoundException("User not found");
  }

  const isPasswordValid = await user.comparePassword(loginDto.password);
  if (!isPasswordValid) {
    throw new UnauthorizedException("Invalid email/password");
  }

  const { token, expiresAt } = signJwtToken({ userId: user.id });

  return {
    user: user.omitPassword(),
    accessToken: token,
    expiresAt,
  };
};

async function generateUsername(name) {
  const cleanName = name.replace(/\s+/g, "").toLowerCase();
  const baseUsername = cleanName;

  const uuidSuffix = uuidv4().replace(/\s+/g, "").slice(0, 4);

  let username = `${baseUsername}${uuidSuffix}`;
  let existingUser = await UserModel.findOne({ username });

  while (existingUser) {
    username = `${baseUsername}${uuidv4().replace(/\s+/g, "").slice(0, 4)}`;
    existingUser = await UserModel.findOne({ username });
  }

  return username;
}
