import { plainToInstance } from "class-transformer";
import { validate } from "class-validator";
import { HTTPSTATUS } from "../config/http.config.js";
import { ErrorCodeEnum } from "../enums/error-code.enum.js";
import { asyncHandler } from "./asyncHandler.middeware.js";

export function asyncHandlerAndValidation(
  dto,
  source = "body",
  handler
) {
  return asyncHandler(withValidation(dto, source)(handler));
}

export function withValidation(
  DtoClass,
  source = "body"
) {
  return function (
    handler
  ) {
    return async (req, res, next) => {
      try {
        const dtoInstance = plainToInstance(DtoClass, req[source]);
        const errors = await validate(dtoInstance);

        if (errors.length > 0) {
          return formatValidationError(res, errors);
        }

        return handler(req, res, dtoInstance);
      } catch (error) {
        next(error);
      }
    };
  };
}

function formatValidationError(res, errors) {
  return res.status(HTTPSTATUS.BAD_REQUEST).json({
    message: "Validation failed",
    errorCode: ErrorCodeEnum.VALIDATION_ERROR,
    errors: errors.map((err) => ({
      field: err.property,
      message: err.constraints,
    })),
  });
}
