import mongoose, { Schema } from "mongoose";
import { v4 as uuidv4 } from "uuid";
import { hashValue, compareValue } from "../../utils/bcrypt.js";

const UserSchema = new Schema(
  {
    _id: { type: String, default: uuidv4 },
    name: { type: String, required: true },
    username: { type: String, required: true, unique: true },
    email: { type: String, required: true, unique: true },
    password: { type: String, required: true },
    imageUrl: { type: String },
  },
  {
    timestamps: true,
    toJSON: {
      virtuals: true,
      transform: function (doc, ret) {
        ret.id = ret._id;
        delete ret._id;
        delete ret.__v;
      },
    },
    toObject: { virtuals: true },
  }
);

// Virtuals for relationships
UserSchema.virtual("events", {
  ref: "Event",
  localField: "_id",
  foreignField: "user",
});

UserSchema.virtual("integrations", {
  ref: "Integration",
  localField: "_id",
  foreignField: "user",
});

UserSchema.virtual("availability", {
  ref: "Availability",
  localField: "_id",
  foreignField: "user",
  justOne: true,
});

UserSchema.virtual("meetings", {
  ref: "Meeting",
  localField: "_id",
  foreignField: "user",
});

UserSchema.pre("save", async function () {
  const user = this;

  if (!this.isModified("password")) {
    return;
  }

  user.password = await hashValue(user.password);
});

UserSchema.methods.comparePassword = async function (
  candidatePassword
) {
  return compareValue(candidatePassword, this.password);
};

UserSchema.methods.omitPassword = function () {
  const userObject = this.toObject();
  delete userObject.password;
  delete userObject.__v;
  // Ensure id is present
  userObject.id = userObject._id;
  delete userObject._id;
  return userObject;
};

export const UserModel = mongoose.model("User", UserSchema);
