import { HTTPSTATUS } from "../config/http.config.js";
import { asyncHandlerAndValidation } from "../middlewares/withValidation.middleware.js";
import {
  CreateEventDto,
  EventIdDTO,
  UpdateEventDto,
  UserNameAndSlugDTO,
  UserNameDTO,
} from "../database/dto/event.dto.js";
import {
  createEventService,
  deleteEventService,
  getPublicEventByUsernameAndSlugService,
  getPublicEventsByUsernameService,
  getUserEventsService,
  toggleEventPrivacyService,
  updateEventService,
} from "../services/event.service.js";
import { asyncHandler } from "../middlewares/asyncHandler.middeware.js";

export const createEventController = asyncHandlerAndValidation(
  CreateEventDto,
  "body",
  async (req, res, createEventDto) => {
    const userId = req.user?.id;

    const event = await createEventService(userId, createEventDto);

    return res.status(HTTPSTATUS.CREATED).json({
      message: "Event created successfully",
      event,
    });
  }
);

export const getUserEventsController = asyncHandler(
  async (req, res) => {
    const userId = req.user?.id;
    const { events, username } = await getUserEventsService(userId);

    return res.status(HTTPSTATUS.OK).json({
      message: "User event fetched successfully",
      data: {
        events,
        username,
      },
    });
  }
);

export const toggleEventPrivacyController = asyncHandlerAndValidation(
  EventIdDTO,
  "body",
  async (req, res, eventIdDto) => {
    const userId = req.user?.id;

    const event = await toggleEventPrivacyService(userId, eventIdDto.eventId);

    return res.status(HTTPSTATUS.OK).json({
      message: `Event set to ${
        event.isPrivate ? "private" : "public"
      } successfully`,
    });
  }
);

export const getPublicEventsByUsernameController = asyncHandlerAndValidation(
  UserNameDTO,
  "params",

  async (req, res, userNameDto) => {
    const { user, events } = await getPublicEventsByUsernameService(
      userNameDto.username
    );

    return res.status(HTTPSTATUS.OK).json({
      message: "Public events fetched successfully",
      user,
      events,
    });
  }
);

export const getPublicEventByUsernameAndSlugController =
  asyncHandlerAndValidation(
    UserNameAndSlugDTO,
    "params",
    async (req, res, userNameAndSlugDto) => {
      const event = await getPublicEventByUsernameAndSlugService(
        userNameAndSlugDto
      );

      return res.status(HTTPSTATUS.OK).json({
        message: "Event details fetched successfully",
        event,
      });
    }
  );

export const deleteEventController = asyncHandlerAndValidation(
  EventIdDTO,
  "params",
  async (req, res, eventIdDto) => {
    const userId = req.user?.id;

    await deleteEventService(userId, eventIdDto.eventId);
    return res.status(HTTPSTATUS.OK).json({
      message: "Event deleted successfully",
    });
  }
);

export const updateEventController = asyncHandlerAndValidation(
  UpdateEventDto,
  "body",
  async (req, res, updateEventDto) => {
    const userId = req.user?.id;
    const eventId = req.params.eventId;

    const event = await updateEventService(userId, eventId, updateEventDto);

    return res.status(HTTPSTATUS.OK).json({
      message: "Event updated successfully",
      event,
    });
  }
);
