import "reflect-metadata";
import "dotenv/config";
import "./src/config/passport.config.js";
import express from "express";
import cors from "cors";
import fs from "fs";
import http from "http";
import https from "https";
import { config } from "./src/config/app.config.js";
import { HTTPSTATUS } from "./src/config/http.config.js";
import { errorHandler } from "./src/middlewares/errorHandler.middleware.js";
import { initializeDatabase } from "./src/database/database.js";
import authRoutes from "./src/routes/auth.route.js";
import passport from "passport";
import eventRoutes from "./src/routes/event.route.js";
import availabilityRoutes from "./src/routes/availability.route.js";
import integrationRoutes from "./src/routes/integration.route.js";
import meetingRoutes from "./src/routes/meeting.route.js";
import morgan from "morgan";

const app = express();
const BASE_PATH = config.BASE_PATH;

app.use(express.json());
app.use(express.urlencoded({ extended: true }));
app.use(passport.initialize());

const corsOptions = {
  origin: ["http://localhost:5173", "https://calendar.clikkle.com", "https://admin.clikkle.com"],
  credentials: true,
  methods: ["GET", "POST", "PUT", "DELETE", "OPTIONS"],
  allowedHeaders: [
    "Content-Type",
    "Authorization",
    "org-token",
    "X-Requested-With",
    "Accept",
    "Origin",
  ],
};

app.use(cors(corsOptions));
app.options("*", cors(corsOptions));
app.use(morgan("dev"));

app.get("/", async (req, res) => {
  res.status(HTTPSTATUS.OK).json({
    message: "Hello From the Clikkle Calendar Back",
  });
});

app.use(`${BASE_PATH}/auth`, authRoutes);
app.use(`${BASE_PATH}/event`, eventRoutes);
app.use(`${BASE_PATH}/availability`, availabilityRoutes);
app.use(`${BASE_PATH}/integration`, integrationRoutes);
app.use(`${BASE_PATH}/meeting`, meetingRoutes);

app.use(errorHandler);

const port = process.env.PORT;

const privateKeyPath = "../ssl/keys/cf24b_85237_149ac01eadff64cae5ebbb907eba6d3b.key";
const certificatePath = "../ssl/certs/www_api_calendar_clikkle_com_cf24b_85237_1777375597_0d57d721e88a85df37ada69c9cc44fa5.crt";

let server;

try {
  const privateKey = fs.readFileSync(privateKeyPath, "utf8");
  const certificate = fs.readFileSync(certificatePath, "utf8");
  const credentials = { key: privateKey, cert: certificate };

  server = https.createServer(credentials, app);
  server.listen(port, async () => {
    await initializeDatabase();
    console.log(`HTTPS Server running on port ${port} in ${config.NODE_ENV}`);
  });
} catch (err) {
  console.warn("SSL certificates not found, falling back to HTTP:", err.message);
  
  server = http.createServer(app);
  server.listen(port, async () => {
    await initializeDatabase();
    console.log(`HTTP Server running on port ${port} in ${config.NODE_ENV}`);
  });
}

export default app;